<?php
/**
 * Handles the CSS functions of the cache, minification and combining files
 *
 * @link       http://vipercache.com
 * @since      1.0.0
 *
 * @package    ViperCache
 * @subpackage ViperCache/includes
 */

	class CYB_Css_Utilities extends CYB_base_Utilities {
		public $html = "";
		private $tags = array();
		private $exceptLinks = "";
		private $stopRenderBlockCSS = 'N';

		public function __construct($html, $stopRenderBlockCSS) {
			$this->html = $html;
			$this->stopRenderBlockCSS = $stopRenderBlockCSS;
			$this->set_exceptLinks_tags();
			$this->get_tags();
			$this->tags_reorder();
      $this->cssmin = new CYB_CSSMin();
		}

		public function combineCss(){
			$all = array();
			$group = array();

			foreach ($this->tags as $key => $value) {

				if(preg_match("/<link/i", $value["text"])){

					if($this->exceptLinks){
						if(strpos($this->exceptLinks, $value["text"]) !== false){
							array_push($all, $group);
							$group = array();
							continue;
						}
					}

					if(!$this->checkInternal($value["text"])){
						array_push($all, $group);
						$group = array();
						continue;
					}

					if(count($group) > 0){
						if($group[0]["media"] == $value["media"]){
							array_push($group, $value);
						}else{
							array_push($all, $group);
							$group = array();
							array_push($group, $value);
						}
					}else{
						array_push($group, $value);
					}

					if($value === end($this->tags)){
						array_push($all, $group);
					}
				}

				if(preg_match("/<style/i", $value["text"])){
					if(count($group) > 0){
						array_push($all, $group);
						$group = array();
					}
				}
			}

			if(count($all) > 0){
				$all = array_reverse($all);

				foreach ($all as $group_key => $group_value) {
					if(count($group_value) > 0){

						$combined_css = "";
						$combined_name = $this->create_name($group_value);
						$combined_link = "";

						$cachFilePath = $cacheDir = CYB_Viper_Cache::getCachePath() . 'minified/'.$combined_name;
						$cssLink = CYB_Viper_Cache::getCacheURL() . 'minified/'.$combined_name;

						if(is_dir($cachFilePath)){
							if($cssFiles = @scandir($cachFilePath, 1)){

								if($this->stopRenderBlockCSS == 'N') {
									$combined_link = '<link rel="stylesheet" type="text/css" href="'.$cssLink."/".$cssFiles[0].'" media="'.$group_value[0]["media"].'"/>';
								}
								else {
									$combined_link = '<link rel="preload" href="'.$cssLink."/".$cssFiles[0].'" as="style" onload="this.onload=null;this.rel=\'stylesheet\'" media="'.$group_value[0]["media"].'" ><noscript><link rel="stylesheet" href="'.$cssLink."/".$cssFiles[0].'"></noscript>';
								}

								if($css_content = $this->read_file($cssLink."/".$cssFiles[0])){
									$combined_link = $this->to_inline($combined_link, $css_content);
								}
							}
						}
            else{
							$combined_css = $this->create_content(array_reverse($group_value));
							$combined_css = $this->fix_charset($combined_css);

							if($combined_css) {

								$this->createFolder($cachFilePath, $combined_css, 'css', time());

								if(is_dir($cachFilePath)){
									if($cssFiles = @scandir($cachFilePath, 1)) {

										if($this->stopRenderBlockCSS == 'N') {
											$combined_link = '<link rel="stylesheet" type="text/css" href="'.$cssLink."/".$cssFiles[0].'" media="'.$group_value[0]["media"].'"/>';
										}
										else {
											$combined_link = '<link rel="preload" href="'.$cssLink."/".$cssFiles[0].'" as="style" onload="this.onload=null;this.rel=\'stylesheet\'" media="'.$group_value[0]["media"].'" ><noscript><link rel="stylesheet" href="'.$cssLink."/".$cssFiles[0].'"></noscript>';
										}
										$combined_link = $this->to_inline($combined_link, $combined_css);
									}
								}
							}
						}

						if($combined_link){
							foreach (array_reverse($group_value) as $tag_key => $tag_value) {
								$text = substr($this->html, $tag_value["start"], ($tag_value["end"]-$tag_value["start"] + 1));

								if($tag_key > 0){
									$this->html = substr_replace($this->html, "<!-- ".$text." -->", $tag_value["start"], ($tag_value["end"] - $tag_value["start"] + 1));
								}else{
									$this->html = substr_replace($this->html, "<!-- ".$text." -->"."\n".$combined_link, $tag_value["start"], ($tag_value["end"] - $tag_value["start"] + 1));
								}
							}
						}
					}
				}
			}

			return $this->html;
		}

		public function create_content($group_value){
			$combined_css = "";
			foreach ($group_value as $tag_key => $tag_value) {
				$minifiedCss = $this->minify($tag_value["href"]);

				if($minifiedCss){
					$combined_css = $minifiedCss["cssContent"].$combined_css;
				}else{
					return false;
				}
			}

			return $combined_css;
		}

		public function create_name($arr){
			$name = "";
			foreach ($arr as $tag_key => $tag_value) {
				$name = $name.$this->remove_query_string($tag_value["href"]);
			}
			return md5($name);
		}

		public function to_inline($link, $css_content){

			if(!preg_match("/\smedia\=[\'\"]all[\'\"]/i", $link)){
				return $link;
			}

			// If css content size is > 14kb then best served as a file
			if(strlen($css_content) > 14000) {
				return $link;
			}

			$link = "<style>".$css_content."</style>";

			return $link;
		}

		public function tags_reorder(){
		    $sorter = array();
		    $ret = array();

		    foreach ($this->tags as $ii => $va) {
		        $sorter[$ii] = $va['start'];
		    }

		    asort($sorter);

		    foreach ($sorter as $ii => $va) {
		        $ret[$ii] = $this->tags[$ii];
		    }

		    $this->tags = $ret;
		}

    // Find <link> tags in comments, scripts and noscript
		public function set_exceptLinks_tags(){
			$comment_tags = $this->find_tags("<!--", "-->");

			foreach ($comment_tags as $key => $value) {
				$this->exceptLinks = $value["text"].$this->exceptLinks;
			}

			// does html contains <noscript> tag
			if(preg_match("/<noscript/i", $this->html)){
				$noscript_tags = $this->find_tags("<noscript", "</noscript>");

				foreach ($noscript_tags as $key => $value) {
					$this->exceptLinks = $value["text"].$this->exceptLinks;
				}
			}

      // find links added via script
			$script_tags = $this->find_tags("<script", "</script>");

			foreach ($script_tags as $key => $value) {
				$link_tags = $this->find_tags("<link", ">", $value["text"]);

				if(count($link_tags) > 0){
					$this->exceptLinks = $value["text"].$this->exceptLinks;
				}
			}
		}

		public function get_tags(){
			$style_tags = $this->find_tags("<style", "</style>");
			$this->tags = array_merge($this->tags, $style_tags);

			$link_tags = $this->find_tags("<link", ">");

			foreach ($link_tags as $key => $value) {
				if(preg_match("/avada-dynamic-css-css/", $value["text"])){
					continue;
				}

				preg_match("/media\=[\'\"]([^\'\"]+)[\'\"]/", $value["text"], $media);
				preg_match("/href\=[\'\"]([^\'\"]+)[\'\"]/", $value["text"], $href);

				$media[1] = (isset($media[1]) && $media[1]) ? trim($media[1]) : "";
				$value["media"] = (isset($media[1]) && $media[1]) ? $media[1] : "all";

				if(isset($href[1])){
					$href[1] = trim($href[1]);
					$value["href"] = (isset($href[1]) && $href[1]) ? $href[1] : "";

					if(preg_match("/href\s*\=/i", $value["text"])){
						if(preg_match("/rel\s*\=\s*[\'\"]\s*stylesheet\s*[\'\"]/i", $value["text"])){
							array_push($this->tags, $value);
						}
					}
				}
			}
		}

		public function remove_query_string($url){
			$url = preg_replace("/^(\/\/|http\:\/\/|https\:\/\/)(www\.)?/", "", $url);
			$url = preg_replace("/\?.*/", "", $url);

			return $url;
		}

		public function minify($url){
			$this->url = $url;
			$md5 = md5($this->remove_query_string($url));

			$cachFilePath = CYB_Viper_Cache::getCachePath() . 'minified/'.$md5;
			$cssLink = CYB_Viper_Cache::getCacheURL() . 'minified/'.$md5;

			if(is_dir($cachFilePath)) {
				if($cssFiles = @scandir($cachFilePath, 1)){
					if($cssContent = $this->file_get_contents_curl($cssLink."/".$cssFiles[0])){
						return array("cachFilePath" => $cachFilePath, "cssContent" => $cssContent, "url" => $cssLink."/".$cssFiles[0], "realUrl" => $url);
					}else{
						return false;
					}
				}
			}
      else {
				if($cssContent = $this->file_get_contents_curl($url, "?v=".time())) {

					$original_content_length = strlen($cssContent);
          if($ipath = CYB_Viper_Cache::getPath($url)) {
            $cssContent = $this->files2Bse64($ipath,$cssContent);
          }
          $cssContent = $this->cssmin->_process($cssContent);
					$cssContent = $this->fixPathsInCssContent($cssContent, $url);
					$cssContent = str_replace("\xEF\xBB\xBF", '', $cssContent);

					// If the content is empty, the file is not created. This breaks "combine css" feature
					if(strlen($cssContent) == 0 && $original_content_length > 0){
						return array("cssContent" => "", "url" => $url);
					}

					if(!is_dir($cachFilePath)){
						$prefix = time();
						$this->createFolder($cachFilePath, $cssContent, 'css', $prefix);
					}

					if($cssFiles = @scandir($cachFilePath, 1)){
						return array("cachFilePath" => $cachFilePath, "cssContent" => $cssContent, "url" => $cssLink."/".$cssFiles[0], "realUrl" => $url);
					}
				}
			}
			return false;
		}

		public function fixPathsInCssContent($css, $url){
			$this->url_for_fix = $url;

			$css = preg_replace("/@import\s+[\"\']([^\;\"\'\)]+)[\"\'];/", "@import url($1);", $css);
			$css = preg_replace_callback("/url\(([^\)\n]*)\)/", array($this, 'newImgPath'), $css);
			$css = preg_replace_callback('/@import\s+url\(([^\)]+)\);/i', array($this, 'fix_import_rules'), $css);
			$css = $this->fix_charset($css);

			return $css;
		}

		public function newImgPath($matches){
			$matches[1] = trim($matches[1]);

			if(preg_match("/data\:image\/svg\+xml/", $matches[1])){
				$matches[1] = $matches[1];
			}else{
				$matches[1] = str_replace(array("\"","'"), "", $matches[1]);
				$matches[1] = trim($matches[1]);

				if(!$matches[1]){
					$matches[1] = "";
				}else if(preg_match("/^(\/\/|http|\/\/fonts|data:image|data:application)/", $matches[1])){
					if(preg_match("/fonts\.googleapis\.com/", $matches[1])){ // for safari browser
						$matches[1] = '"'.$matches[1].'"';
					}else{
						$matches[1] = $matches[1];
					}
				}else if(preg_match("/^\//", $matches[1])){
					$homeUrl = str_replace(array("http:", "https:"), "", home_url());
					$matches[1] = $homeUrl.$matches[1];
				}else if(preg_match("/^\.\/.+/i", $matches[1])){
					//$matches[1] = str_replace("./", get_template_directory_uri()."/", $matches[1]);
					$matches[1] = str_replace("./", dirname($this->url_for_fix)."/", $matches[1]);
				}else if(preg_match("/^(?P<up>(\.\.\/)+)(?P<name>.+)/", $matches[1], $out)){
					$count = strlen($out["up"])/3;
					$url = dirname($this->url);
					for($i = 1; $i <= $count; $i++){
						$url = substr($url, 0, strrpos($url, "/"));
					}
					$url = str_replace(array("http:", "https:"), "", $url);
					$matches[1] = $url."/".$out["name"];
				}else{
					$url = str_replace(array("http:", "https:"), "", dirname($this->url));
					$matches[1] = $url."/".$matches[1];
				}
			}

			return "url(".$matches[1].")";
		}

		public function fix_charset($css){
			preg_match_all('/@charset[^\;]+\;/i', $css, $charsets);
			if(count($charsets[0]) > 0){
				$css = preg_replace('/@charset[^\;]+\;/i', "", $css);
				foreach($charsets[0] as $charset){
					$css = $charset."\n".$css;
				}
			}
			return $css;
		}

		public function fix_import_rules($matches){
			if($this->is_internal_css($matches[1])){
				if($cssContent = $this->file_get_contents_curl($matches[1], "?v=".time())){
					$tmp_url = $this->url;
					$this->url = $matches[1];
					$cssContent = $this->fixPathsInCssContent($cssContent, $matches[1]);
					$this->url = $tmp_url;
					return $cssContent;
				}
			}

			return $matches[0];
		}

	  public function checkInternal($link){
			$httpHost = str_replace("www.", "", $_SERVER["HTTP_HOST"]);

			if(preg_match("/href=[\"\'](.*?)[\"\']/", $link, $href)){

				if(preg_match("/^\/[^\/]/", $href[1])){
					return $href[1];
				}

				if(@strpos($href[1], $httpHost)){
					return $href[1];
				}
			}
			return false;
		}

		public function is_internal_css($url){
			$http_host = trim($_SERVER["HTTP_HOST"], "www.");

			$url = trim($url);
			$url = trim($url, "'");
			$url = trim($url, '"');

			$url = str_replace(array("http://", "https://"), "", $url);

			$url = trim($url, '//');
			$url = trim($url, 'www.');

			if($url){
				if(preg_match("/".$http_host."/i", $url)){
					return true;
				}
			}

			return false;
		}

    public function file_get_contents_curl($url, $version = ""){

    	if($data = $this->read_file($url)){
    		return $data;
    	}

  		$url = str_replace('&#038;', '&', $url);

      	if(preg_match("/\.php\?/i", $url)){
      		$version = "";
  		}

  		if(preg_match("/(fonts\.googleapis\.com|iire-social-icons)/i", $url)){
  			$version = "";
  			$url = str_replace(array("'",'"'), "", $url);
  		}

      	$url = $url.$version;

  		if(preg_match("/^\/[^\/]/", $url)){
  			$url = get_option("home").$url;
  		}

  		if(preg_match("/http\:\/\//i", home_url())){
  			$url = preg_replace("/^\/\//", "http://", $url);
  		}else if(preg_match("/https\:\/\//i", home_url())){
  			$url = preg_replace("/^\/\//", "https://", $url);
  		}

  		$response = wp_remote_get($url, array('timeout' => 10, 'user-agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_2) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/49.0.2623.110 Safari/537.36'));

  		if ( !$response || is_wp_error( $response ) ) {
  			return false;
  		}else{
  			if(wp_remote_retrieve_response_code($response) == 200){
  				$data = wp_remote_retrieve_body( $response );

  				if(preg_match("/\<\!DOCTYPE/i", $data) || preg_match("/<\/\s*html\s*>/i", $data)){
  					return false;
  				}else if(!$data){
  					return "/* empty */";
  				}else{
  					return $data;
  				}
  			}else if(wp_remote_retrieve_response_code($response) == 404){
  				if(preg_match("/\.css/", $url)){
  					return "/*404*/";
  				}else{
  					return "<!-- 404 -->";
  				}
  			}
  		}
  	}

    /**
     * Import files into the CSS, base64-ized.
     *
     * @url(image.jpg) images will be loaded and their content merged into the
     * original file, to save HTTP requests.
     *
     * @param string $source  The file to import files for
     * @param string $content The CSS content to import files for
     *
     * @return string
     */
    protected function files2Bse64($source, $content) {
      $importExtensions = array(
        'gif' => 'data:image/gif',
        'png' => 'data:image/png',
        'jpe' => 'data:image/jpeg',
        'jpg' => 'data:image/jpeg',
        'jpeg' => 'data:image/jpeg',
        'svg' => 'data:image/svg+xml',
        'woff' => 'data:application/x-font-woff',
        'tif' => 'image/tiff',
        'tiff' => 'image/tiff',
        'xbm' => 'image/x-xbitmap',
      );

      $regex = '/url\((["\']?)(.+?)\\1\)/i';
      if (preg_match_all($regex, $content, $matches, PREG_SET_ORDER)) {
          $search = array();
          $replace = array();

          // loop the matches
          foreach ($matches as $match) {

              $extension = substr(strrchr($match[2], '.'), 1);
              if ($extension && !array_key_exists($extension, $importExtensions)) {
                  continue;
              }

              // get the path for the file that will be imported
              $path = $match[2];
              $path = dirname($source).'/'.$path;

              // only replace the import with the content if we're able to get
              // the content of the file, and it's relatively small
              if ($this->canReadFile($path) && $this->canReadBySize($path)) {

                  // grab content && base64-ize
                  $importContent = $this->read_file($path);
                  $importContent = base64_encode($importContent);

                  // build replacement
                  $search[] = $match[0];
                  $replace[] = 'url('.$importExtensions[$extension].';base64,'.$importContent.')';
              }
          }

          // replace the import statements
          $content = str_replace($search, $replace, $content);
      }

      return $content;
    }
	}

class CYB_CSSMin {

    /**
     * Minify CSS.
     *
     * @param string $css CSS to be minified
     *
     * @return string
     */
    public static function minify($css)
    {
        $cssmin = new CYB_CSSMin();
        return $cssmin->_process($css);
    }

    /**
     * Are we "in" a hack? I.e. are some browsers targetted until the next comment?
     *
     * @var bool
     */
    protected $_inHack = false;

    /**
     * Minify a CSS string
     *
     * @param string $css
     *
     * @return string
     */
    public function _process($css)
    {
        // $css = str_replace("\r\n", "\n", $css);
        $css = str_replace(array("\r\n", "\n", "\t"), '', $css);

        // preserve empty comment after '>'
        // http://www.webdevout.net/css-hacks#in_css-selectors
        $css = preg_replace('@>/\\*\\s*\\*/@', '>/*keep*/', $css);

        // preserve empty comment between property and value
        // http://css-discuss.incutio.com/?page=BoxModelHack
        $css = preg_replace('@/\\*\\s*\\*/\\s*:@', '/*keep*/:', $css);
        $css = preg_replace('@:\\s*/\\*\\s*\\*/@', ':/*keep*/', $css);

        // apply callback to all valid comments (and strip out surrounding ws
        $css = preg_replace_callback('@\\s*/\\*([\\s\\S]*?)\\*/\\s*@'
            ,array($this, '_commentCB'), $css);
        // remove ws around { } and last semicolon in declaration block
        $css = preg_replace('/\\s*{\\s*/', '{', $css);
        $css = preg_replace('/;?\\s*}\\s*/', '}', $css);

        // remove ws surrounding semicolons
        $css = preg_replace('/\\s*;\\s*/', ';', $css);

        // remove ws around urls
        $css = preg_replace('/
                url\\(      # url(
                \\s*
                ([^\\)]+?)  # 1 = the URL (really just a bunch of non right parenthesis)
                \\s*
                \\)         # )
            /x', 'url($1)', $css);

        // remove ws between rules and colons
        $css = preg_replace('/
                \\s*
                ([{;])              # 1 = beginning of block or rule separator
                \\s*
                ([\\*_]?[\\w\\-]+)  # 2 = property (and maybe IE filter)
                \\s*
                :
                \\s*
                (\\b|[#\'"-])        # 3 = first character of a value
            /x', '$1$2:$3', $css);

        // remove ws in selectors
        $css = preg_replace_callback('/
                (?:              # non-capture
                    \\s*
                    [^~>+,\\s]+  # selector part
                    \\s*
                    [,>+~]       # combinators
                )+
                \\s*
                [^~>+,\\s]+      # selector part
                {                # open declaration block
            /x'
            ,array($this, '_selectorsCB'), $css);

        // minimize hex colors
        // $css = $this->shortenHexColorCodes($css);
        $css = preg_replace('/([^=])#([a-f\\d])\\2([a-f\\d])\\3([a-f\\d])\\4([\\s;\\}])/i', '$1#$2$3$4$5', $css);

        // Shorten font weights
        // $css = $this->shortenFontWeights($css);

        // remove spaces between font families
        $css = preg_replace_callback('/font-family:([^;}]+)([;}])/'
            ,array($this, '_fontFamilyCB'), $css);

        $css = preg_replace('/@import\\s+url/', '@import url', $css);

        // replace any ws involving newlines with a single newline
        $css = preg_replace('/[ \\t]*\\n+\\s*/', "\n", $css);

        // separate common descendent selectors w/ newlines (to limit line lengths)
        // $css = preg_replace('/([\\w#\\.\\*]+)\\s+([\\w#\\.\\*]+){/', "$1\n$2{", $css);

        // Use newline after 1st numeric value (to limit line lengths).
        // $css = preg_replace('/
        //     ((?:padding|margin|border|outline):\\d+(?:px|em)?) # 1 = prop : 1st numeric value
        //     \\s+
        //     /x'
        //     ,"$1\n", $css);

        // prevent triggering IE6 bug: http://www.crankygeek.com/ie6pebug/
        $css = preg_replace('/:first-l(etter|ine)\\{/', ':first-l$1 {', $css);

        // Strip Empty Tags
        $css = preg_replace('/(?<=^)[^\{\};]+\{\s*\}/', '', $css);
        $css = preg_replace('/(?<=(\}|;))[^\{\};]+\{\s*\}/', '', $css);

        // Remove double spaces
        $css = preg_replace('/\s\s+/', ' ', $css);

        return trim($css);
    }

    /**
     * Replace what looks like a set of selectors
     *
     * @param array $m regex matches
     *
     * @return string
     */
    protected function _selectorsCB($m)
    {
        // remove ws around the combinators
        return preg_replace('/\\s*([,>+~])\\s*/', '$1', $m[0]);
    }

    /**
     * Process a comment and return a replacement
     *
     * @param array $m regex matches
     *
     * @return string
     */
    protected function _commentCB($m)
    {
        $hasSurroundingWs = (trim($m[0]) !== $m[1]);
        $m = $m[1];
        // $m is the comment content w/o the surrounding tokens,
        // but the return value will replace the entire comment.
        if ($m === 'keep') {
            return '/**/';
        }
        if ($m === '" "') {
            // component of http://tantek.com/CSS/Examples/midpass.html
            return '/*" "*/';
        }
        if (preg_match('@";\\}\\s*\\}/\\*\\s+@', $m)) {
            // component of http://tantek.com/CSS/Examples/midpass.html
            return '/*";}}/* */';
        }
        if ($this->_inHack) {
            // inversion: feeding only to one browser
            if (preg_match('@
                    ^/               # comment started like /*/
                    \\s*
                    (\\S[\\s\\S]+?)  # has at least some non-ws content
                    \\s*
                    /\\*             # ends like /*/ or /**/
                @x', $m, $n)) {
                // end hack mode after this comment, but preserve the hack and comment content
                $this->_inHack = false;
                return "/*/{$n[1]}/**/";
            }
        }
        if (substr($m, -1) === '\\') { // comment ends like \*/
            // begin hack mode and preserve hack
            $this->_inHack = true;
            return '/*\\*/';
        }
        if ($m !== '' && $m[0] === '/') { // comment looks like /*/ foo */
            // begin hack mode and preserve hack
            $this->_inHack = true;
            return '/*/*/';
        }
        if ($this->_inHack) {
            // a regular comment ends hack mode but should be preserved
            $this->_inHack = false;
            return '/**/';
        }
        // Issue 107: if there's any surrounding whitespace, it may be important, so
        // replace the comment with a single space
        return $hasSurroundingWs // remove all other comments
            ? ' '
            : '';
    }

    /**
     * Process a font-family listing and return a replacement
     *
     * @param array $m regex matches
     *
     * @return string
     */
    protected function _fontFamilyCB($m)
    {
        // Issue 210: must not eliminate WS between words in unquoted families
        $pieces = preg_split('/(\'[^\']+\'|"[^"]+")/', $m[1], null, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
        $out = 'font-family:';
        while (null !== ($piece = array_shift($pieces))) {
            if ($piece[0] !== '"' && $piece[0] !== "'") {
                $piece = preg_replace('/\\s+/', ' ', $piece);
                $piece = preg_replace('/\\s?,\\s?/', ',', $piece);
            }
            $out .= $piece;
        }
        return $out . $m[2];
    }

    /**
     * Shorthand hex color codes.
     * #FF0000 -> #F00.
     *
     * @param string $content The CSS content to shorten the hex color codes for
     *
     * @return string
     */
    protected function shortenHexColorCodes($content) {

        $content = preg_replace('/(?<=[: ])#([0-9a-z])\\1([0-9a-z])\\2([0-9a-z])\\3(?=[; }])/i', '#$1$2$3', $content);

        // we can shorten some even more by replacing them with their color name
        $colors = array(
            '#F0FFFF' => 'azure',
            '#F5F5DC' => 'beige',
            '#A52A2A' => 'brown',
            '#FF7F50' => 'coral',
            '#FFD700' => 'gold',
            '#808080' => 'gray',
            '#008000' => 'green',
            '#4B0082' => 'indigo',
            '#FFFFF0' => 'ivory',
            '#F0E68C' => 'khaki',
            '#FAF0E6' => 'linen',
            '#800000' => 'maroon',
            '#000080' => 'navy',
            '#808000' => 'olive',
            '#CD853F' => 'peru',
            '#FFC0CB' => 'pink',
            '#DDA0DD' => 'plum',
            '#800080' => 'purple',
            '#F00' => 'red',
            '#FA8072' => 'salmon',
            '#A0522D' => 'sienna',
            '#C0C0C0' => 'silver',
            '#FFFAFA' => 'snow',
            '#D2B48C' => 'tan',
            '#FF6347' => 'tomato',
            '#EE82EE' => 'violet',
            '#F5DEB3' => 'wheat',
        );

        return preg_replace_callback(
            '/(?<=[: ])('.implode(array_keys($colors), '|').')(?=[; }])/i',
            function ($match) use ($colors) {
                return $colors[strtoupper($match[0])];
            },
            $content
        );
    }

    /**
     * Shorten CSS font weights.
     *
     * @param string $content The CSS content to shorten the font weights for
     *
     * @return string
     */
    protected function shortenFontWeights($content)
    {
        $weights = array(
            'normal' => 400,
            'bold' => 700,
        );

        $callback = function ($match) use ($weights) {
            return $match[1].$weights[$match[2]];
        };

        return preg_replace_callback('/(font-weight\s*:\s*)('.implode('|', array_keys($weights)).')(?=[;}])/', $callback, $content);
    }
}
