<?php

/**
 * The Viper Cache Caching Engine
 *
 * @link       http://vipercache.com
 * @since      1.0.0
 *
 * @package    ViperCache
 * @subpackage ViperCache/includes
 */

require_once "class-base-utilities.php";
require_once "class-css-utilities.php";
require_once "class-js-utilities.php";
require_once "class-html-utilities.php";

/**
 * The Viper Cahce Caching Engine
 *
 * @since      1.0.0
 * @package    ViperCache
 * @subpackage ViperCache/includes
 * @author     John Merrick | WP Tookkit
 */

class CYB_Viper_Cache {

	private static $cacheSettings;

	private static $fileCache = array();
	private static $cacheStats = array();
	private static $disable_cache_uris = array();
	private static $cacheClearedThisRequest = false;
	private static $clearScheduledThisRequest = false;
	private static $lastRecursiveDeleteError = false;
  private static $preloadScheduledThisRequest = false;
  private static $viperCacheRejectUri = '';

	public function setupCaching($cacheSettings) {

		self::$cacheSettings = $cacheSettings;

    $this->set_wp_content_url();

    if(self::$cacheSettings['active'] != 'Y') {
			return; // cache is not active
		}

		if(is_admin()) {
			add_action('publish_post', 'CYB_Viper_Cache::action_publishPost');
			add_action('publish_page', 'CYB_Viper_Cache::action_publishPost');
      add_action('save_post', 'CYB_Viper_Cache::action_publishPost');
      //'clean_object_term_cache', 'clean_post_cache', 'clean_term_cache', 'clean_page_cache',
			foreach(array('after_switch_theme', 'customize_save_after', 'activated_plugin', 'deactivated_plugin', 'update_option_sidebars_widgets') as $action) {
				add_action($action, 'CYB_Viper_Cache::action_clearPageCache'); // Schedules a cache clear for immediately so it won't lag current request.
		  }

      add_filter('widget_update_callback', 'CYB_Viper_Cache::filter_widgetUpdateCallback'); // Schedules a cache clear for immediately so it won't lag current request.

			if($_SERVER['REQUEST_METHOD'] == 'POST') {
				foreach(array(
					'/\/wp\-admin\/options\.php$/',
					'/\/wp\-admin\/options\-permalink\.php$/'
					) as $pattern) {
					if(preg_match($pattern, $_SERVER['REQUEST_URI'])) {
						self::scheduleCacheClear();
						break;
					}
				}
			}
		}
    else {
      add_filter( 'wp_resource_hints', 'CYB_Viper_Cache::_extra_preconnect', 10, 2 );
      add_filter( 'viper_cache_excluded_js_files', array($this, 'update_excluded_js_files'));
    }

		add_action('comment_post', 'CYB_Viper_Cache::action_commentPost'); // Might not be logged in
		add_filter('wp_redirect', 'CYB_Viper_Cache::redirectFilter');

    // Get the list of cache disabled URI's
    self::$disable_cache_uris = array_flip(get_option('vipercache-disable-cache-uris', array() ));

		// Routines to clear cache if our URL is hit with a POST, PUT, DELETE or any other non 'GET' request, then clear that page.
		$file = self::fileFromRequest( ($_SERVER['HTTP_HOST'] ? $_SERVER['HTTP_HOST'] : $_SERVER['SERVER_NAME']), $_SERVER['REQUEST_URI']);
		$fileDeleted = false;
		$doDelete = false;
		if($_SERVER['REQUEST_METHOD'] != 'GET') {
			$doDelete = true;
		}

		if($doDelete) {
			@unlink($file);
			$fileDeleted = true;
		}

		if(self::isCachable()) {
			if( (! $fileDeleted) ) { // Then serve the file if it's still valid
				$stat = @stat($file);
				if($stat) {
					$age = time() - $stat[9];
					if($age < 10000) {
						readfile($file); // sends file to stdout
						die();
					}
				}
			}

			ob_start(array($this, 'obComplete')); // Setup routine to store the file
		}
	}

  public static function updateSettings($settings) {
    self::$cacheSettings = $settings;
  }

	public static function redirectFilter($status) {
		if(! defined('VICA_NOCACHE')) {
			define('VICA_NOCACHE', true);
		}
		return $status;
	}

	public static function isCachable() {

    // If you want to tell Viper Cache not to cache something in another plugin, simply define one of these.
		if((defined('VICA_NOCACHE') && VICA_NOCACHE)) {
			return false;
		}

		if(self::$cacheSettings['cacheHTTPS'] == 'N') {
			if(self::isHTTPSPage()) {
				return false;
			}
		}

		if(is_admin()) { return false; } // do not cache any admin pages.

    if ( strstr( $_SERVER['REQUEST_URI'], 'robots.txt' ) || strstr( $_SERVER['REQUEST_URI'], '.htaccess' ) ) {
    	return false;
    }

    // Don't cache Ajax requests
    if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
    	return false;
    }

    // Don't cache customizer preview.
    if ( isset( $_POST['wp_customize'] ) ) {
	     return;
     }

		$uri = $_SERVER['REQUEST_URI'];

		if(strrpos($uri, '/') !== strlen($uri) - 1) { // must end with a '/' char.
			return false;
		}
		if($_SERVER['REQUEST_METHOD'] != 'GET') { return false; } // Only cache GET's

    // Don't cache query strings unless they are /?123132423=123123234 DDoS style.
		if(isset($_SERVER['QUERY_STRING']) && strlen($_SERVER['QUERY_STRING']) > 0 && (! preg_match('/^\d+=\d+$/', $_SERVER['QUERY_STRING'])) ) {
			return false;
		}

		// Wordpress login cookies indicates user logged in
		if(is_array($_COOKIE)) {
			foreach(array_keys($_COOKIE) as $c) {
				foreach(array('comment_author','wp-postpass','wordpress_logged_in','wptouch_switch_toggle','wpmp_switcher') as $b) {
					if(strpos($c, $b) !== false) { return false; } // Have a cookie so user must not be cached
				}
			}
		}

    // See if cache is disabled at a post/page level
    if ( isset( self::$disable_cache_uris[ $uri ] ) ) {
      return false;
    }

    // Get the list or uri to not cache
    self::$viperCacheRejectUri = self::get_reject_uri(self::$cacheSettings['doNotCacheUrls']);

    // See if uri is set to do not cache
    if ( isset( self::$viperCacheRejectUri ) ) {
       $request_uri = explode( '?', $_SERVER['REQUEST_URI'] );
       $request_uri = reset( ( $request_uri ) );
       if(preg_match( '#^(' . self::$viperCacheRejectUri . ')$#', $request_uri ) ) {
    	    return false;
       }
    }

		return true;
	}

	public static function isHTTPSPage() {
		if( isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] && $_SERVER['HTTPS'] != 'off') {
			return true;
		}
    // In case we're behind a proxy and user used HTTPS.
		if( !empty( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' ) {
			return true;
		}

		return false;
	}

  public static function isHTTPSSite() {
    return 'https' === parse_url( home_url(), PHP_URL_SCHEME );
  }

	public function obComplete($buffer = '') {

		if(function_exists('is_404') && is_404()) {
			return $buffer;
		}

    if(function_exists('is_feed') && is_feed()) {
			return $buffer;
		}

    // On wp login page
		if($GLOBALS["pagenow"] == "wp-login.php") {
			return $buffer . "<!-- WP-Login pages are not cached by Viper Cache -->";
		}
    else if(isset($_GET["preview"])) {
				return $buffer . "<!-- Preview pages are not cached by Viper Cache -->";
    }

	/**
	 * Hack to allow overriding of the DONOTCACHEPAGE behavior.
	 * To warn conflict with some plugins like Thrive Leads.
   */
   $overrideDonotcachepage = apply_filters( 'viper_cache_override_donotcachepage', false );

		// These constants might have been set after we did the initial isCachable check by e.g. wp_redirect filter. So if they are now set just return the buffer and don't cache.
		if((defined('VICA_NOCACHE') && VICA_NOCACHE) || $overrideDonotcachepage) {
				return $buffer;
		}

    // If web page is less than 1000 bytes, don't cache it.
		if(strlen($buffer) < 1000) {
			return $buffer . "<!-- Page Size (".$origSize." bytes) to small to be Cached by Viper Cache -->";
		}

		$file = self::fileFromRequest( ($_SERVER['HTTP_HOST'] ? $_SERVER['HTTP_HOST'] : $_SERVER['SERVER_NAME']), $_SERVER['REQUEST_URI']);
		self::makeDirIfNeeded($file);

    $htmlOptions = array();

    if(self::$cacheSettings['fixImgDim'] == 'Y') {
      $buffer = $this->fix_image_dimensions($buffer);
    }

    if(self::$cacheSettings['combineCSS'] == 'Y') {
      $cssUtil = new CYB_Css_Utilities($buffer, self::$cacheSettings['stopRenderBlockCSS']);
      $buffer = $cssUtil->combineCss();
      $htmlOptions = array_merge($htmlOptions, array('cssMinifier' => 'CYB_CSSMin::minify'));
    }

    if(self::$cacheSettings['combineJS'] == 'Y') {
      $jsUtil = new CYB_JS_Utilities($buffer, self::$cacheSettings['deferJS']);
      $buffer = $jsUtil->combineJs();
      $htmlOptions = array_merge($htmlOptions, array('jsMinifier' => 'CYB_JSMin::minify'));
    }

    $buffer = $this->combine_google_fonts($buffer);

    // Try and reduce the size of the HTML
    $origSize = strlen($buffer);
    $buffer = CYB_HTML_Utilities::minify($buffer, $htmlOptions);
    $minifiedSize = strlen($buffer);

		$append = "\n<!-- Cached by Viper Cache ";
		$append .= "Created: " . date('Y-m-d H:i:s T') . ". ";
		$append .= "HTTPS page: " . (self::isHTTPSPage() ? 'Yes' : 'No') . ". ";
		$append .= "Size: original=" . $origSize . " bytes. minified=" . $minifiedSize . " bytes.";
		$append .= " -->";

		// Insert Critical CSS
		if ( !empty(self::$cacheSettings['criticalCSS']) ) {
			$buffer = preg_replace( '/<head(.*)>/U', "<head$1><style id='vc-critical-css'>".self::$cacheSettings['criticalCSS']."</style>", $buffer, 1 );
		}

		@file_put_contents($file, $buffer . $append, LOCK_EX);
		@chmod($file, 0644);

    $stats = json_encode(Array('size' => array('original' => $origSize,  'minified' => $minifiedSize)));
    $file = str_replace('.html', '.stats', $file);
		@file_put_contents($file, $stats, LOCK_EX);
		@chmod($file, 0644);

		return $buffer . $append;
	}

	public function combine_google_fonts($content) {
		// Get all Google Fonts CSS files.
		$content_sans_comments = preg_replace( '/<!--(.*)-->/Uis', '', $content );
		preg_match_all( '/<link(?:\s+(?:(?!href\s*=\s*)[^>])+)?(?:\s+href\s*=\s*([\'"])((?:https?:)?\/\/fonts\.googleapis\.com\/css(?:(?!\1).)+)\1)(?:\s+[^>]*)?>/iU', $content_sans_comments, $matches );

		if ( ! $matches[2] || 1 === count( $matches ) ) {
			return $content;
		}

		$fonts   = array();
		$subsets = array();

		foreach ( $matches[2] as $k => $font ) {
			// Get fonts name.
			$font = str_replace( array( '%7C', '%7c' ), '|', $font );
			$font = explode( 'family=', $font );
			$font = ( isset( $font[1] ) ) ? explode( '&', $font[1] ) : array();

			// Add font to the collection.
			$fonts = array_merge( $fonts, explode( '|', reset( $font ) ) );

			// Add subset to collection.
			$subset = ( is_array( $font ) ) ? end( $font ) : '';
			if ( false !== strpos( $subset, 'subset=' ) ) {
				$subset  = explode( 'subset=', $subset );
				$subsets = array_merge( $subsets, explode( ',', $subset[1] ) );
			}

			// Remove the current Google Font links
			$content = str_replace( $matches[0][ $k ], '', $content );
		}

		// Combine fonts tag.
		$subsets = ( $subsets ) ? '&subset=' . implode( ',', array_filter( array_unique( $subsets ) ) ) : '';
		$fonts   = implode( '|', array_filter(array_unique( $fonts )));
		$fonts   = str_replace( '|', '%7C', $fonts );

		if ( ! empty( $fonts ) ) {
			$fonts  = '<link rel="stylesheet" href="https://fonts.googleapis.com/css?family=' . $fonts . $subsets . '" />';
			$content = preg_replace( '/<head(.*)>/U', '<head$1>' . $fonts, $content, 1 );
		}

		return $content;

	}

  public function fix_image_dimensions($content) {

    // Find all image tags
    preg_match_all( '/<img[^>]+>/i', $content, $images );

    // If there are no images, return
    if ( count( $images ) < 1 ) {
        return $content;
    }

      foreach ( $images[0] as $image ) {

          // Match all image attributes
          $attributes = 'src|srcset|longdesc|alt|class|id|usemap|align|border|hspace|vspace|crossorigin|ismap|sizes|width|height|style';
          preg_match_all( '/(' . $attributes . ')=("[^"]*")/i', $image, $img );

          // If image has a 'src', continue
          if ( ! in_array( 'src', $img[1] ) ) {
              continue;
          }

          $widthIdx = array_search( 'width', $img[1] );
          $heightIdx = array_search( 'height', $img[1] );

          // If no 'width' or 'height' is available or blank, calculate dimensions
          if ( $widthIdx === false || $heightIdx === false || ( $widthIdx !== false && $img[2][$widthIdx] == '""' ) || ( $heightIdx !== false && $img[2][$heightIdx] == '""' ) ) {

            // Split up string of attributes into variables
            $attributes = explode( '|', $attributes );
            foreach ( $attributes as $variable ) {
                ${$variable} = in_array( $variable, $img[1] ) ? ' ' . $variable . '=' . $img[2][array_search( $variable, $img[1] )] : '';
            }
            $src = $img[2][array_search( 'src', $img[1] )];
            // If image is an SVG/WebP with no dimensions, set specific dimensions
            if ( preg_match( '/(.*).svg|.webp/i', $src ) ) {
                if ( ! in_array( 'width', $img[1] ) || ! in_array( 'height', $img[1] ) || ( in_array( 'width', $img[1] ) && in_array( '""', $img[2] ) ) || ( in_array( 'height', $img[1] ) && in_array( '""', $img[2] ) ) ) {
                    $width = '100%';
                    $height = 'auto';
                }
            }
            // Else, get accurate width and height attributes
            else {
                // Check if source is relative URL and convert to absolute if it is
                if(preg_match('/^"(\.\.|\.|\/)\//', $src)) {
                  $absSrc = preg_replace('/^"(\.\.|\.|\/)\//', '', $src);
                  $absSrc = trailingslashit($_SERVER['DOCUMENT_ROOT']).$absSrc;
                  list( $width, $height ) = getimagesize( str_replace( "\"", "" , $absSrc ) );
                }
                else {
									$aSrc = str_replace( "\"", "" , $src );
									if(empty($aSrc)) {
										$width = "";
										$height = "";
									}
									else {
                  	list( $width, $height ) = getimagesize( $aSrc );
									}
                }
                if(empty($width) && empty($height)) continue;
            }

            // See if image is a WP Optimiser Lazy Load types
             if(stripos($image, 'data-wpopti-lazyimg-src') !== false) {
               $srcLabel = 'data-wpopti-lazyimg-src';
             }
             else {
               $srcLabel = 'src';
             }
            // Recreate the image tag with dimensions set
            $tag = sprintf( '<img %s=%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s width="%s" height="%s">',  $srcLabel, $src, $style, $srcset, $longdesc, $alt, $class, $id, $usemap, $align, $border, $hspace, $vspace, $crossorigin, $ismap, $sizes, $width, $height );
            $content = str_replace( $image, $tag, $content );
          }
      }
      // Return all image with dimensions
      return $content;
  }

	public static function fileFromRequest($host, $URI) {
		return self::fileFromURI($host, $URI, self::isHTTPSPage());
	}

	public static function fileFromURI($host, $URI, $isHTTPS) {
		$key = $host . $URI . ($isHTTPS ? '_HTTPS' : '');
		if(isset(self::$fileCache[$key])) { return self::$fileCache[$key]; }
		$host = preg_replace('/[^a-zA-Z0-9\-\.]+/', '', $host);
		$URI = preg_replace('/(?:[^a-zA-Z0-9\-\_\.\~\/]+|\.{2,})/', '', $URI); //Strip out bad chars and multiple dots
		if(preg_match('/\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)(.*)$/', $URI, $matches)) {
			$URI = $matches[1] . '/';
			for($i = 2; $i <= 6; $i++) {
				$URI .= strlen($matches[$i]) > 0 ? $matches[$i] : '';
				$URI .= $i < 6 ? '_' : '';
			}
		}
		$ext = '';
		if($isHTTPS) { $ext = '_https'; }
		$file = self::getCachePath() . $host . '_' . $URI . '_vipercache' . $ext . '.html';
		self::$fileCache[$key] = $file;
		return $file;
	}

	public static function makeDirIfNeeded($file) {
		$file = preg_replace('/\/[^\/]*$/', '', $file);
		if(! is_dir($file)) {
			@mkdir($file, 0755, true);
		}
	}

	public static function cacheDirectoryTest() {
		$cacheDir = self::getCachePath();
		if(! is_dir($cacheDir)) {
			if(! @mkdir($cacheDir, 0755, true)) {
				$err = error_get_last();
				$msg = "The directory $cacheDir does not exist and we could not create it.";
				if($err) {
					$msg .= ' The error we received was: ' . $err['message'];
				}
				return $msg;
			}
		}
		if(! @file_put_contents($cacheDir . 'test.php', 'test')) {
			$err = error_get_last();
			$msg = "We could not write to the file $cacheDir" . "test.php when testing if the cache directory is writable.";
			if($err) {
				$msg .= " The error was: " . $err['message'];
			}
			return $msg;
		}

		return false;
	}

	public static function action_publishPost($id) {
		$postPerm = get_permalink($id);
    $homeURL = trailingslashit(get_option("home"));
		self::deleteFileFromPermalink($postPerm);
    self::deleteFileFromPermalink($homeURL);

    $urls = array();
    set_transient( 'CYB_Viper_Cache_Preload', 'On', 180 );
	  // The home Page
		array_push($urls, array("url" => $postPerm));
    array_push($urls, array("url" => $homeURL));
    // See if using static Pages
    if(get_option('show_on_front') == 'page'){
      $pof = get_option('page_on_front');
      $pfp = get_option('page_for_posts');
      if($pof > 0) {
    		$pofPerm = get_permalink($pof);
    		self::deleteFileFromPermalink($postPerm);
        array_push($urls, array("url" => $pofPerm));
      }
      if($pfp > 0) {
    		$pofPerm = get_permalink($pfp);
    		self::deleteFileFromPermalink($postPerm);
        array_push($urls, array("url" => $pofPerm));
      }
    }

    $viperpreload_background_process = $GLOBALS['CYB_Vipercache_background_process'];

		foreach ($urls as $key => $arr) {
      $viperpreload_background_process->push_to_queue( $arr["url"]);
		}

    $viperpreload_background_process->save()->dispatch();
	}

	public static function action_commentPost($commentID) {
		$c = get_comment($commentID, ARRAY_A);
		$perm = get_permalink($c['comment_post_ID']);
		self::deleteFileFromPermalink($perm);
		// self::scheduleCacheClear();
	}

	public static function action_clearPageCache() { // Can safely call this as many times as we like because it'll only schedule one clear
		self::scheduleCacheClear();
	}

	public static function filter_clearPageCache($param) { // Can safely call this as many times as we like because it'll only schedule one clear
		self::scheduleCacheClear();
    return $param;
	}

  public static function filter_widgetUpdateCallback($instance) {
    self::scheduleCacheClear();

    return $instance;
  }

  /**
   * When Woocommerce, EDD, iThemes Exchange, Jigoshop & WP-Shop options are saved or deleted,
   * update the List of uri not to cache
   *
   * @since 2.0.0
   *
   * @param array $old_value An array of previous settings values.
   * @param array $new_value An array of submitted settings values.
   */
  function after_update_single_options( $old_value, $new_value ) {

  	if ( $old_value !== $_newvalue ) {
	     CYB_Viper_Cache::update_3rdparty_reject_uri();
  	}
  }

  /**
   * When EDD, iThemes Exchange, Jigoshop & WP-Shop options are saved or deleted,
   * update the List of uri not to cache
   *
   * @since 2.0.0
   *
   * @param array $old_value An array of previous settings values.
   * @param array $new_value An array of submitted settings values.
   */
  function after_update_array_options( $old_value, $new_value ) {

  	$options = array(
  		'purchase_page',
  		'jigoshop_cart_page_id',
  		'jigoshop_checkout_page_id',
  		'jigoshop_myaccount_page_id',
  	);

  	foreach ( $options as $val ) {
  		if ( (!isset( $old_value[$val] ) && isset($new_value[$val]) ) ||
         ( isset($old_value[$val], $new_value[$val] ) && $old_value[$val] !== $new_value[$val] ) ) {
  	    CYB_Viper_Cache::update_3rdparty_reject_uri();
  			break;
  		}
  	}
  }

	public static function scheduleCacheClear() {
		if(self::$clearScheduledThisRequest) { return; }
		self::$clearScheduledThisRequest = true;
    wp_clear_scheduled_hook( 'vipercache_cache_clear' );
		wp_schedule_single_event(time() - 15, 'vipercache_cache_clear');
    self::initiate_cron();
	}

	public static function scheduledCacheClear() {

		self::clearPageCacheSafe(); // Will only run if clearPageCache() has not run this request
	}

	public static function deleteFileFromPermalink($perm) {
		if(preg_match('/\/\/([^\/]+)(\/.*)$/', $perm, $matches)) {
			$host = $matches[1];
			$uri = $matches[2];
			$file = self::fileFromRequest($host, $uri);
			if(is_file($file)) {
				@unlink($file);
			}
		}
	}

	public static function getPath($url) {
		if(preg_match('/\/\/([^\/]+)(\/.*)$/', $url, $matches)) {
			$host = $matches[1];
			$uri = $matches[2];
			$file = untrailingslashit($_SERVER['DOCUMENT_ROOT']) . $uri;
			if(is_file($file)) {
				return $file;
			}
		}

    return false;
	}

	public static function getCachePath($root = false) {

  	if ( ! is_multisite() || $root ) {
  		$path = WP_CONTENT_DIR . '/uploads/vipercache/';
  	}
    else {
      // Current blog infos
	    $blog_id  = get_current_blog_id();

      $path = WP_CONTENT_DIR . '/uploads/vipercache/' . $blog_id . '/';
    }

    return $path;
  }

	public static function getCacheURL($root = false) {

  	if ( ! is_multisite() || $root ) {
  		$url = str_replace(array("http:", "https:"), "", CYBVC_WP_CONTENT_URL).'/uploads/vipercache/';
  	}
    else {
      // Current blog infos
	    $blog_id  = get_current_blog_id();

      $url = str_replace(array("http:", "https:"), "", CYBVC_WP_CONTENT_URL).'/uploads/vipercache/' . $blog_id .'/';
    }

    return $url;
  }

  public static function start_cache_preload() {

    if(self::is_cache_preloading()) return;

    $urls = array();

    set_transient( 'CYB_Viper_Cache_Preload', 'On', 300 );

	  // The home Page
		array_push($urls, array("url" => get_option("home")));

    // Latest 5 posts
		$recent_posts = wp_get_recent_posts(array(
		    'numberposts' => 5,
			  'orderby' => 'post_date',
			  'order' => 'DESC',
			  'post_type' => 'post',
			  'post_status' => 'publish',
        'meta_query' => array(
          'relation' => 'OR',
            array(
             'key' => 'vipercache-disable-cache-uris',
             'compare' => 'NOT EXISTS'
            ),
            array(
               'key'     => 'vipercache-disable-cache-uris',
               'value'   => 'N',
               'compare' => '!='
            ) ),
			   'suppress_filters' => true));

    foreach( $recent_posts as $recent ){
      array_push($urls, array("url" => get_permalink($recent["ID"])));
    }

    // Latest 5 pages
		$recent_posts = wp_get_recent_posts(array(
		    'numberposts' => 5,
			  'orderby' => 'post_date',
			  'order' => 'DESC',
			  'post_type' => 'page',
			  'post_status' => 'publish',
        'meta_query' => array(
          'relation' => 'OR',
            array(
             'key' => 'vipercache-disable-cache-uris',
             'compare' => 'NOT EXISTS'
            ),
            array(
               'key'     => 'vipercache-disable-cache-uris',
               'value'   => 'N',
               'compare' => '!='
            ) ),
			   'suppress_filters' => true));

    foreach( $recent_posts as $recent ){
      array_push($urls, array("url" => get_permalink($recent["ID"])));
    }

    // Latest 5 WP Easy Pages pages
		$recent_posts = wp_get_recent_posts(array(
		    'numberposts' => 5,
			  'orderby' => 'post_date',
			  'order' => 'DESC',
			  'post_type' => 'wpeo_optinpage',
			  'post_status' => 'publish',
        'meta_query' => array(
          'relation' => 'OR',
            array(
             'key' => 'vipercache-disable-cache-uris',
             'compare' => 'NOT EXISTS'
            ),
            array(
               'key'     => 'vipercache-disable-cache-uris',
               'value'   => 'N',
               'compare' => '!='
            ) ),
			   'suppress_filters' => true));

    foreach( $recent_posts as $recent ){
      array_push($urls, array("url" => get_permalink($recent["ID"])));
    }

    $viperpreload_background_process = $GLOBALS['CYB_Vipercache_background_process'];

		foreach ($urls as $key => $arr) {
      $viperpreload_background_process->push_to_queue( $arr["url"]);
		}

    $viperpreload_background_process->save()->dispatch();

	}

  public static function is_cache_preloading() {

    return get_transient( 'CYB_Viper_Cache_Preload' );
  }

  public static function cancel_preload() {

    $viperpreload_background_process = $GLOBALS['CYB_Vipercache_background_process'];

    $viperpreload_background_process->cancel_process();

    // Wait for the preload to cancel or timeout
    $seconds = 0;
    while($viperpreload_background_process->is_process_running() && $seconds++ < 10) {
        sleep(1);
    }

    delete_transient( 'CYB_Viper_Cache_Preload' );
  }

	public static function getCacheStats() {
		self::$cacheStats = array(
			'files' => 0,
			'dirs' => 0,
			'data' => 0,
			'compressedBytes' => 0,
			'uncompressedFiles' => 0,
			'uncompressedBytes' => 0,
			'oldestFile' => false,
			'newestFile' => false,
			'largestFile' => 0,
			);
		self::recursiveStats(self::getCachePath());
		return self::$cacheStats;
	}

	private static function recursiveStats($dir) {
		$files = @array_diff(@scandir($dir), array('.','..'));
    if(is_null($files)) return;
		foreach($files as $file) {
			$fullPath = $dir . '/' . $file;

			if(is_dir($fullPath)) {
				self::$cacheStats['dirs']++;
				self::recursiveStats($fullPath);
			} else {
				if($file == 'clear.lock' || $file == 'index.html' || $file == 'htaccess.backup' || $file == 'test.php') { continue; }
        if(preg_match('/.+\.stats$/', $file)) { continue; }
        if(preg_match('/.+\.css$/', $file)) { continue; }
        if(preg_match('/.+\.js$/', $file)) { continue; }

        $statsFile = $dir . '/' . str_replace('.html', '.stats', $file);
        $fileStats = @file_get_contents($statsFile);
        $fileStats = json_decode($fileStats);

				self::$cacheStats['files']++;
				$stat = @stat($fullPath);

				if(is_array($stat)) {
					$size = $fileStats->size->original;
					if($size) {
						self::$cacheStats['uncompressedFiles']++;
						self::$cacheStats['uncompressedBytes'] += $fileStats->size->original;
					  self::$cacheStats['compressedBytes'] += $fileStats->size->minified;

						if(self::$cacheStats['largestFile'] < $size) {
							self::$cacheStats['largestFile'] = $size;
						}
					}

					$ctime = $stat[10];
					if(self::$cacheStats['oldestFile'] > $ctime || self::$cacheStats['oldestFile'] === false) {
						self::$cacheStats['oldestFile'] = $ctime;
					}
					if(self::$cacheStats['newestFile'] === false || self::$cacheStats['newestFile'] < $ctime) {
						self::$cacheStats['newestFile'] = $ctime;
					}
				}
			}
		}
	}

	public static function clearPageCacheSafe() {
		if(self::$cacheClearedThisRequest) { return; }
		self::$cacheClearedThisRequest = true;

    if(self::is_cache_preloading()) {
      self::cancel_preload();
    }

		self::clearPageCache();
	}

	public static function clearPageCache() { //If a clear is in progress this does nothing.
		self::$cacheStats = array(
			'dirsDeleted' => 0,
			'filesDeleted' => 0,
			'totalData' => 0,
			'totalErrors' => 0,
			'error' => '',
			);
		$cacheClearLock = self::getCachePath() .'clear.lock';
		if(! is_file($cacheClearLock)) {
			if(! @touch($cacheClearLock)) {
				self::$cacheStats['error'] = "Could not create a lock file $cacheClearLock to clear the cache.";
				self::$cacheStats['totalErrors']++;
				return self::$cacheStats;
			}
		}
		$fp = fopen($cacheClearLock, 'w');
		if(! $fp) {
			self::$cacheStats['error'] = "Could not open the lock file $cacheClearLock to clear the cache. Please make sure the directory is writable by your web server.";
			self::$cacheStats['totalErrors']++;
			return self::$cacheStats;
		}
		if(flock($fp, LOCK_EX | LOCK_NB)) { //non blocking exclusive flock attempt. If we get a lock then it continues and returns true. If we don't lock, then return false, don't block and don't clear the cache.
					// This logic means that if a cache clear is currently in progress we don't try to clear the cache.
					// This prevents web server children from being queued up waiting to be able to also clear the cache.
			self::$lastRecursiveDeleteError = false;
			self::recursiveDelete(self::getCachePath());
			if(self::$lastRecursiveDeleteError) {
				self::$cacheStats['error'] = self::$lastRecursiveDeleteError;
				self::$cacheStats['totalErrors']++;
			}
			flock($fp, LOCK_UN);
		}
		fclose($fp);

    wp_clear_scheduled_hook( 'vipercache_cache_clear' );

		return self::$cacheStats;
	}

	public static function recursiveDelete($dir) {
		$files = array_diff(@scandir($dir), array('.','..'));
		foreach ($files as $file) {
			if(is_dir($dir . '/' . $file)) {
				if(! self::recursiveDelete($dir . '/' . $file)) {
					return false;
				}
			} else {
				if($file == 'clear.lock') { continue; } //Don't delete our lock file
				if($file == 'htaccess.backup') { continue; } //Don't delete our htaccess backup file
				$size = @filesize($dir . '/' . $file);
				if($size) {
					self::$cacheStats['totalData'] += round($size / 1024);
				}
				if(strpos($dir, 'vipercache/') === false) {
					self::$lastRecursiveDeleteError = "Not deleting file in directory $dir because it appears to be in the wrong path.";
					self::$cacheStats['totalErrors']++;
					return false; //Safety check that we're in a subdir of the cache
				}
				if(@unlink($dir . '/' . $file)) {
					self::$cacheStats['filesDeleted']++;
				} else {
					self::$lastRecursiveDeleteError = "Could not delete file " . $dir . "/" . $file . " : " . self::getLastError();
					self::$cacheStats['totalErrors']++;
					return false;
				}
			}
		}
		if($dir != self::getCachePath()) {
			if(strpos($dir, 'vipercache/') === false) {
				self::$lastRecursiveDeleteError = "Not deleting directory $dir because it appears to be in the wrong path.";
				self::$cacheStats['totalErrors']++;
				return false; //Safety check that we're in a subdir of the cache
			}
			if(@rmdir($dir)) {
				self::$cacheStats['dirsDeleted']++;
			} else {
				self::$lastRecursiveDeleteError = "Could not delete directory $dir : " . self::getLastError();
				self::$cacheStats['totalErrors']++;
				return false;
			}
			return true;
		} else {
			return true;
		}
	}

	public static function addHtaccessCode($action, $settingsUpdate = true) {

    if($settingsUpdate && is_multisite()) return false;

		if($action != 'add' && $action != 'remove') {
			die("Error: addHtaccessCode must be called with 'add' or 'remove' as param");
		}

    $htaccessBackupFile = self::getCachePath(true) . 'htaccess.backup';
		$htaccessPath = self::getHtaccessPath();
		if(! $htaccessPath) {
			return "Viper Cache could not find your .htaccess file.";
		}
		$fh = @fopen($htaccessPath, 'r+');
		if(! $fh) {
			$err = error_get_last();
			return $err['message'];
		}
		flock($fh, LOCK_EX);
		fseek($fh, 0, SEEK_SET); //start of file
		clearstatcache();
		$contents = fread($fh, filesize($htaccessPath));
		if(! $contents) {
			fclose($fh);
			return "Could not read from $htaccessPath";
		}
		$contents = preg_replace('/#VIPERCACHESITECODE.*VIPERCACHESITECODE[r\s\n\t]*/s', '', $contents);
		if($action == 'add') {
      @file_put_contents($htaccessBackupFile, $contents, LOCK_EX);
			$code = self::getHtaccessCode();
			$contents = $code . "\n" . $contents;
		}
		ftruncate($fh, 0);
		fseek($fh, 0, SEEK_SET);
		fwrite($fh, $contents);
		flock($fh, LOCK_UN);
		fclose($fh);
		return false;
	}

	public static function getHtaccessCode() {
		$siteURL = site_url();
		$homeURL = home_url();
		$pathPrefix = "";

		if(preg_match('/^https?:\/\/[^\/]+\/(.+)$/i', $siteURL, $matches)) {
			$path = $matches[1];
			$path = preg_replace('/^\//', '', $path);
			$path = preg_replace('/\/$/', '', $path);
			$pathPrefix = '/' . $path; // Which is: /my/path
		}

		$matchCaps = '$1/$2_$3_$4_$5_$6';
		if(preg_match('/^https?:\/\/[^\/]+\/(.+)$/i', $homeURL, $matches)) {
			$path = $matches[1];
			$path = preg_replace('/^\//', '', $path);
			$path = preg_replace('/\/$/', '', $path);
			$pieces = explode('/', $path);
			if(count($pieces) == 1) {
				// No path:       "/wp-content/vipercache/%{HTTP_HOST}_$1/$2_$3_$4_$5_$6_vipercache%{ENV:WRDFNC_HTTPS}.html%{ENV:WRDFNC_ENC}" [L]
				// One path:  "/mdm/wp-content/vipercache/%{HTTP_HOST}_mdm/$1_$2_$3_$4_$5_vipercache%{ENV:WRDFNC_HTTPS}.html%{ENV:WRDFNC_ENC}" [L]
				$matchCaps = $pieces[0] . '/$1_$2_$3_$4_$5';
			} else if(count($pieces) == 2) {
				$matchCaps = $pieces[0] . '/' . $pieces[1] . '/$1_$2_$3_$4';
			} else {
				$matchCaps = '$1/$2_$3_$4_$5_$6'; // defaults to the regular setting but this won't work. However user should already have gotten a warning that we don't support sites more than 2 dirs deep.
			}
		}

		$sslString = "RewriteCond %{HTTPS} off";
		if(self::$cacheSettings['cacheHTTPS'] == 'Y') {
			$sslString = "";
		}

    // Hotlinking protection
    $blogUrl = strtolower( get_bloginfo( 'url' ) );
    $blogUrl = str_replace( array('https://','http://','www.'),'',$blogUrl );
    $hlImg = plugin_dir_url( __FILE__ ) . 'images/hotlink-disabled.jpeg';

    // Get the correct root for RewriteBase
    $home_root = parse_url($homeURL);
		if ( isset( $home_root['path'] ) )
			$home_root = trailingslashit($home_root['path']);
		else
			$home_root = '/';

    $imgHotlink = "";
		if(!is_multisite() && self::$cacheSettings['fixImgHL'] == 'Y') {
			$imgHotlink = <<<EOT
    <IfModule mod_rewrite.c>
      # Hotlink Protection START
      RewriteEngine On
      RewriteBase {$home_root}
    	RewriteCond %{HTTP_REFERER} !^$
    	RewriteCond %{HTTP_REFERER} !^http(s)?://(www\.)?{$blogUrl} [NC]
    	RewriteCond %{HTTP_REFERER} !^http(s)?://(www\.)?google.com [NC]
    	RewriteCond %{HTTP_REFERER} !^http(s)?://(www\.)?bing.com [NC]
    	RewriteCond %{HTTP_REFERER} !^http(s)?://(www\.)?yahoo.com [NC]
    	RewriteRule \.(jpg|jpeg|png|gif|svg)$ {$hlImg} [NC,R,L]
    	# Hotlink Protection END
    </IfModule>
EOT;
    }

    $cacheCode = "";
    if(!is_multisite()) {
    $cacheCode = <<<EOT
    <IfModule mod_rewrite.c>
    	#Prevents garbled chars in cached files if there is no default charset.
    	AddDefaultCharset utf-8

    	#Cache rules:
    	RewriteEngine On
    	RewriteBase {$home_root}
    	RewriteCond %{HTTPS} on
    	RewriteRule .* - [E=WRDFNC_HTTPS:_https]
    	RewriteCond %{REQUEST_METHOD} !=POST
    	{$sslString}
    	RewriteCond %{QUERY_STRING} ^(?:\d+=\d+)?$
    	RewriteCond %{REQUEST_URI} (?:\/|\.html)$ [NC]
    	RewriteCond %{HTTP_COOKIE} !(comment_author|wp\-postpass|wordpress_logged_in|wptouch_switch_toggle|wpmp_switcher|wp_woocommerce_session|safirmobilswitcher=mobil) [NC]
    	RewriteCond %{HTTP_USER_AGENT} !(facebookexternalhit|WhatsApp|Mediatoolkitbot)
    	RewriteCond %{REQUEST_URI} \/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)(.*)$
    	RewriteCond "%{DOCUMENT_ROOT}{$pathPrefix}/wp-content/uploads/vipercache/%{HTTP_HOST}_%1/%2_%3_%4_%5_%6_vipercache%{ENV:WRDFNC_HTTPS}.html%{ENV:WRDFNC_ENC}" -f
    	RewriteRule \/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)\/*([^\/]*)(.*)$ "{$pathPrefix}/wp-content/uploads/vipercache/%{HTTP_HOST}_{$matchCaps}_vipercache%{ENV:WRDFNC_HTTPS}.html%{ENV:WRDFNC_ENC}" [L]
    </IfModule>
EOT;
    }

    $badBotsBlock = "";
		if(!is_multisite() && self::$cacheSettings['blockBots'] == 'Y') {
			$badBotsBlock = <<<EOT
    <IfModule mod_rewrite.c>
      # Bad Bot Blocking START
      RewriteEngine On
      RewriteBase {$home_root}
      RewriteCond %{HTTP_USER_AGENT} ^Aboundex.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^360Spider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^80legs.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Abonti.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Acunetix.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ADmantX.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^AfD\-Verbotsverfahren.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^AhrefsBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^AIBOT.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^AiHitBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Aipbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Alexibot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Alligator.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^AllSubmitter.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Anarchie.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Apexoo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ASPSeek.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Asterias.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Attach.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^autoemailspider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BackDoorBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Backlink\-Ceck.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^backlink\-check.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BacklinkCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BackStreet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BackWeb.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Badass.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bandit.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Barkrowler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BatchFTP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Battleztar\ Bazinga.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BBBike.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BDFetch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BetaBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bigfoot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bitacle.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Blackboard.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Black\ Hole.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BlackWidow.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BLEXBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Blow.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BlowFish.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Boardreader.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bolt.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BotALot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Brandprotect.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Brandwatch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bubing.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Buddy.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BuiltBotTough.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BuiltWith.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Bullseye.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BunnySlippers.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^BuzzSumo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Calculon.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CATExplorador.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CazoodleBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CCBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Cegbfeieh.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CheeseBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CherryPicker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ChinaClaw.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Chlooe.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Claritybot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Cliqzbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Cloud\ mapping.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^coccocbot\-web.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Cogentbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^cognitiveseo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Collector.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^com\.plumanalytics.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Copier.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CopyRightCheck.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Copyscape.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Cosmos.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Craftbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CrazyWebCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Crescent.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^CSHttp.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Curious.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Custo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DatabaseDriverMysqli.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DataCha0s.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DBLBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^demandbase\-bot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Demon.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Deusu.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Devil.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Digincore.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DigitalPebble.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DIIbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Dirbuster.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Disco.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Discobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Discoverybot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DittoSpyder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DomainAppender.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DomainCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DomainSigmaCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DomainStatsBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Dotbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Download\ Wonder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Dragonfly.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Drip.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^DTS\ Agent.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EasyDL.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ebingbong.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^eCatch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ECCP\/1\.0.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ecxi.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EirGrabber.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EMail\ Siphon.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EMail\ Wolf.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EroCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^evc\-batch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Evil.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Exabot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Express\ WebPictures.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ExtLinksBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Extractor.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ExtractorPro.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Extreme\ Picture\ Finder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^EyeNetIE.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ezooms.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^FDM.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^FemtosearchBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^FHscan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Fimap.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Firefox\/7\.0.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^FlashGet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Flunky.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Foobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^fq.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Freeuploader.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^FrontPage.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Fyrebot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GalaxyBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Genieo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GermCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Getintent.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GetRight.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GetWeb.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Gigablast.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Gigabot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^G\-i\-g\-a\-b\-o\-t.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Go\-Ahead\-Got\-It.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Gotit.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GoZilla.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Go\!Zilla.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Grabber.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GrabNet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Grafula.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GrapeFX.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GrapeshotCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GridBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^GT\:\:WWW.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HaosouSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Harvest.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Havij.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HEADMasterSEO.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Heritrix.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Hloader.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HMView.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HTMLparser.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HTTP\:\:Lite.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HTTrack.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Humanlinks.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^HybridBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Iblog.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^IDBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Id\-search.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^IlseBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Image\ Fetch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Image\ Sucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Indy\ Library.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^InfoNaviRobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^InfoTekies.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^instabid.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Intelliseek.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^InterGET.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Internet\ Ninja.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^InternetSeer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^internetVista\ monitor.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ips\-agent.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Iria.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^IRLbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Iskanie.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JamesBOT.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Jbrofuzz.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JennyBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JetCar.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JikeSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JOC\ Web\ Spider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Joomla.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Jorgee.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^JustView.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Jyxobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Kenjin\ Spider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Keyword\ Density.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Kozmosbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Lanshanbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Larbin.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LeechFTP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LeechGet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LexiBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Lftp.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LibWeb.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Libwhisker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Lightspeedsystems.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Likse.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Linkdexbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinkextractorPro.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinkpadBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinkScan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinksManager.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinkWalker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinqiaMetadataDownloaderBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinqiaRSSBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LinqiaScrapeBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Lipperhey.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Litemage\_walker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Lmspider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LNSpiderguy.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ltx71.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^lwp\-request.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^LWP\:\:Simple.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^lwp\-trivial.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Magnet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mag\-Net.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^magpie\-crawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mail\.ru.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Majestic12.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MarkMonitor.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MarkWatch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Masscan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mass\ Downloader.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mata\ Hari.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Meanpathbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^mediawords.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MegaIndex\.ru.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Metauri.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MFC\_Tear\_Sample.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Microsoft\ Data\ Access.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Microsoft\ URL\ Control.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MIDown\ tool.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MIIxpc.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mister\ PiX.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MJ12bot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Mojeek.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Morfeus\ Fucking\ Scanner.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MSFrontPage.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MSIECrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Msrabot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^MS\ Web\ Services\ Client\ Protocol.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^muhstik\-scan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Musobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Name\ Intelligence.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nameprotect.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Navroad.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NearSite.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Needle.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nessus.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NetAnts.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Netcraft.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^netEstate\ NE\ Crawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NetLyzer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NetMechanic.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NetSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nettrack.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Net\ Vampire.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Netvibes.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NetZIP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NextGenSearchBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nibbler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NICErsPRO.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Niki\-bot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nikto.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NimbleCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ninja.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nmap.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^NPbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Nutch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^oBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Octopus.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Offline\ Explorer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Offline\ Navigator.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Openfind.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^OpenLinkProfiler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Openvas.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^OrangeBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^OrangeSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^OutfoxBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PageAnalyzer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Page\ Analyzer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PageGrabber.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^page\ scorer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PageScorer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Panscient.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Papa\ Foto.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pavuk.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^pcBrowser.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PECL\:\:HTTP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PeoplePal.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PHPCrawl.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Picscout.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Picsearch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PictureFinder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pimonster.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pi\-Monster.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pixray.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PleaseCrawl.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^plumanalytics.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pockey.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^POE\-Component\-Client\-HTTP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Probethenet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ProPowerBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ProWebWalker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Psbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Pump.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^PyCurl.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^QueryN\ Metasearch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Qwantify.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankActive.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankActiveLinkBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankFlex.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankingBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankingBot2.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Rankivabot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RankurBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RealDownload.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Reaper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RebelMouse.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Recorder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RedesScrapy.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ReGet.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RepoMonkey.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Ripper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^RocketCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Rogerbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SalesIntelligent.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SBIder.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ScanAlert.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Scanbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Scrapy.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Screaming.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ScreenerBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Searchestate.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SearchmetricsBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Semrush.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SemrushBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SEOkicks.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SEOlyticsCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Seomoz.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SEOprofiler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^seoscanners.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SEOstats.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^sexsearcher.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Seznam.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SeznamBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Siphon.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SISTRIX.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sitebeam.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SiteExplorer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Siteimprove.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SiteLockSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SiteSnagger.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SiteSucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Site\ Sucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sitevigil.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Slackbot\-LinkExpanding.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SlySearch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SmartDownload.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SMTBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Snake.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Snapbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Snoopy.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SocialRankIOBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sogou\ web\ spider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sosospider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sottopop.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SpaceBison.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Spammen.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SpankBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Spanner.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Spbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Spinn3r.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SputnikBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sqlmap.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sqlworm.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sqworm.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Steeler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Stripper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Sucuri.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SuperBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SuperHTTP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Surfbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^SurveyBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Suzuran.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Swiftbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^sysscan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Szukacz.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^T0PHackTeam.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^T8Abot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^tAkeOut.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Teleport.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^TeleportPro.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Telesoft.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Telesphoreo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Telesphorep.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^The\ Intraformant.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^TheNomad.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^TightTwatBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Titan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Toata.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Toweyabot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Trendiction.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Trendictionbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^trendiction\.com.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^trendiction\.de.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^True\_Robot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Turingos.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Turnitin.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^TurnitinBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^TwengaBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Twice.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Typhoeus.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^UnisterBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^URLy\ Warning.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^URLy\.Warning.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Vacuum.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Vagabondo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^VB\ Project.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^VCI.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^VeriCiteCrawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^VidibleScraper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Virusdie.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^VoidEYE.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Voil.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Voltron.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Wallpapers\/3\.0.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WallpapersHD.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WASALive\-Bot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WBSearchBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Webalta.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebAuto.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Auto.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebBandit.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebCollage.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Collage.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebCopier.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WEBDAV.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebEnhancer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Enhancer.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebFetch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Fetch.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebFuck.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Fuck.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebGo\ IS.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebImageCollector.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebLeacher.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebmasterWorldForumBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^webmeup\-crawler.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebPix.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Pix.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebReaper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebSauger.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Sauger.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Webshag.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebsiteExtractor.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebsiteQuester.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Website\ Quester.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Webster.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebStripper.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebSucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Web\ Sucker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebWhacker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WebZIP.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WeSEE.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Whack.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Whacker.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Whatweb.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Who\.is\ Bot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Widow.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WinHTTrack.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WiseGuys\ Robot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WISENutbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Wonderbot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Woobot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Wotbox.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Wprecon.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WPScan.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WWW\-Collector\-E.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WWW\-Mechanize.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WWW\:\:Mechanize.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^WWWOFFLE.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^x09Mozilla.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^x22Mozilla.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Xaldon\ WebSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Xaldon\_WebSpider.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Xenu.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^YoudaoBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Zade.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Zermelo.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Zeus.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^zgrab.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^Zitebot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ZmEu.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ZumBot.* [NC,OR]\nRewriteCond %{HTTP_USER_AGENT} ^ZyBorg.* [NC]
    RewriteRule .* - [F]
    # Bad Bot Blocking END
    </IfModule>
EOT;
    }

		$code = <<<EOT
#VIPERCACHESITECODE - Do not remove the following lines. Disable Web Caching in Viper Cache to remove this data.

<IfModule mod_deflate.c>
  # Force compression for mangled `Accept-Encoding` request headers
  <IfModule mod_setenvif.c>
    <IfModule mod_headers.c>
      SetEnvIfNoCase ^(Accept-EncodXng|X-cept-Encoding|X{15}|~{15}|-{15})$ ^((gzip|deflate)\s*,?\s*)+|[X~-]{4,13}$ HAVE_Accept-Encoding
      RequestHeader append Accept-Encoding "gzip,deflate" env=HAVE_Accept-Encoding
    </IfModule>
  </IfModule>

  # Compress all output labeled with one of the following media types.
  <IfModule mod_filter.c>
    AddOutputFilterByType DEFLATE "application/atom+xml" \
                                  "application/javascript" \
                                  "application/json" \
                                  "application/ld+json" \
                                  "application/manifest+json" \
                                  "application/rdf+xml" \
                                  "application/rss+xml" \
                                  "application/schema+json" \
                                  "application/vnd.geo+json" \
                                  "application/vnd.ms-fontobject" \
                                  "application/x-font-ttf" \
                                  "application/x-javascript" \
                                  "application/x-web-app-manifest+json" \
                                  "application/xhtml+xml" \
                                  "application/xml" \
                                  "font/collection" \
                                  "font/eot" \
                                  "font/opentype" \
                                  "font/otf" \
                                  "font/ttf" \
                                  "image/bmp" \
                                  "image/svg+xml" \
                                  "image/vnd.microsoft.icon" \
                                  "image/x-icon" \
                                  "text/cache-manifest" \
                                  "text/calendar" \
                                  "text/css" \
                                  "text/html" \
                                  "text/javascript" \
                                  "text/plain" \
                                  "text/markdown" \
                                  "text/vcard" \
                                  "text/vnd.rim.location.xloc" \
                                  "text/vtt" \
                                  "text/x-component" \
                                  "text/x-cross-domain-policy" \
                                  "text/xml"
  </IfModule>
  # Map the following filename extensions to the specified
  # encoding type in order to make Apache serve the file types
  # with the appropriate `Content-Encoding` response header
  <IfModule mod_mime.c>
      AddEncoding gzip              svgz
  </IfModule>
</IfModule>

# Remove `ETags` as resources are sent with far-future expires headers.
# `FileETag None` doesn't work in all cases.
<IfModule mod_headers.c>
    Header unset ETag
</IfModule>

FileETag None

# Expires headers
# Serve resources with far-future expires headers.
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresDefault                                      "access plus 1 month"
  # CSS
    ExpiresByType text/css                              "access plus 1 month"
  # Data interchange
    ExpiresByType application/atom+xml                  "access plus 1 hour"
    ExpiresByType application/rdf+xml                   "access plus 1 hour"
    ExpiresByType application/rss+xml                   "access plus 1 hour"
    ExpiresByType application/json                      "access plus 0 seconds"
    ExpiresByType application/ld+json                   "access plus 0 seconds"
    ExpiresByType application/schema+json               "access plus 0 seconds"
    ExpiresByType application/vnd.geo+json              "access plus 0 seconds"
    ExpiresByType application/xml                       "access plus 0 seconds"
    ExpiresByType text/calendar                         "access plus 0 seconds"
    ExpiresByType text/xml                              "access plus 0 seconds"
  # Favicon (cannot be renamed!) and cursor images
    ExpiresByType image/vnd.microsoft.icon              "access plus 1 week"
    ExpiresByType image/x-icon                          "access plus 1 week"
  # HTML
    ExpiresByType text/html                             "access plus 0 seconds"
  # JavaScript
    ExpiresByType application/javascript                "access plus 1 year"
    ExpiresByType application/x-javascript              "access plus 1 year"
    ExpiresByType text/javascript                       "access plus 1 year"
  # Manifest files
    ExpiresByType application/manifest+json             "access plus 1 week"
    ExpiresByType application/x-web-app-manifest+json   "access plus 0 seconds"
    ExpiresByType text/cache-manifest                   "access plus 0 seconds"
  # Markdown
    ExpiresByType text/markdown                         "access plus 0 seconds"
  # Media files
    ExpiresByType audio/ogg                             "access plus 1 month"
    ExpiresByType image/bmp                             "access plus 1 month"
    ExpiresByType image/gif                             "access plus 1 month"
    ExpiresByType image/jpeg                            "access plus 1 month"
    ExpiresByType image/png                             "access plus 1 month"
    ExpiresByType image/svg+xml                         "access plus 1 month"
    ExpiresByType image/webp                            "access plus 1 month"
    ExpiresByType video/mp4                             "access plus 1 month"
    ExpiresByType video/ogg                             "access plus 1 month"
    ExpiresByType video/webm                            "access plus 1 month"
  # Web fonts
    # Collection
    ExpiresByType font/collection                       "access plus 1 month"
    # Embedded OpenType (EOT)
    ExpiresByType application/vnd.ms-fontobject         "access plus 1 month"
    ExpiresByType font/eot                              "access plus 1 month"
    # OpenType
    ExpiresByType font/opentype                         "access plus 1 month"
    ExpiresByType font/otf                              "access plus 1 month"
    # TrueType
    ExpiresByType application/x-font-ttf                "access plus 1 month"
    ExpiresByType font/ttf                              "access plus 1 month"
    # Web Open Font Format (WOFF) 1.0
    ExpiresByType application/font-woff                 "access plus 1 month"
    ExpiresByType application/x-font-woff               "access plus 1 month"
    ExpiresByType font/woff                             "access plus 1 month"
    # Web Open Font Format (WOFF) 2.0
    ExpiresByType application/font-woff2                "access plus 1 month"
    ExpiresByType font/woff2                            "access plus 1 month"
  # Other
    ExpiresByType text/x-cross-domain-policy            "access plus 1 week"
</IfModule>
{$imgHotlink}
{$badBotsBlock}
{$cacheCode}
#Do not remove the above lines. Disable Web caching in Viper Cache to remove this data - VIPERCACHESITECODE
EOT;
		return $code;
	}

	private static function regexSpaceFix($str) {
		return str_replace(' ', '\\s', $str);
	}

	public static function getHtaccessPath() {
		if(file_exists(ABSPATH . '/.htaccess')) {
			return ABSPATH . '/.htaccess';
		}
		if(preg_match('/^https?:\/\/[^\/]+\/?$/i', home_url()) && preg_match('/^https?:\/\/[^\/]+\/.+/i', site_url())) {
			$path = realpath(ABSPATH . '/../.htaccess');
			if(file_exists($path)) {
				return $path;
			}
		}
		return false;
	}

	public static function doNotCache() {
		if(! defined('VICA_NOCACHE')) {
			define('VICA_NOCACHE', true);
		}
	}

	public static function hasLoginCookie() {
  	if(isset($_COOKIE)) {
  		if(is_array($_COOKIE)) {
  			foreach($_COOKIE as $key => $val) {
  				if(strpos($key, 'wordpress_logged_in') == 0) {
  					return true;
  				}
  			}
  		}
  	}
  	return false;
  }

  public static function getLastError() {
		$err = error_get_last();
		if(is_array($err)) {
			return $err['message'];
		}
		return '';
	}

  public static function clean_uri($uri) {
    if ( ! $uri ) {
  		return false;
  	}

  	$path = parse_url( $uri, PHP_URL_PATH );
      return $path;
  }

  public static function clean_url($url) {
  	if ( ! $url ) {
  		return false;
  	}

  	$path = _get_component_from_parsed_url_array(  wp_parse_url( $url ), PHP_URL_PATH );
  	return $path;
  }

  public static function format_filesize_kB( $kiloBytes ) {
    if( ( $kiloBytes / pow( 1024, 4 ) ) > 1) {
        return number_format_i18n( ( $kiloBytes/pow( 1024, 4 ) ), 2 ).'&nbsp;PB';
    } elseif( ( $kiloBytes / pow( 1024, 3 ) ) > 1 ) {
    	return number_format_i18n( ( $kiloBytes/pow( 1024, 3 ) ), 2 ).'&nbsp;TB';
    } elseif( ( $kiloBytes / pow( 1024, 2 ) ) > 1) {
        return number_format_i18n( ( $kiloBytes/pow( 1024, 2 ) ), 2 ).'&nbsp;GB';
    } elseif( ( $kiloBytes / 1024 ) > 1 ) {
        return number_format_i18n( $kiloBytes/1024, 2 ).'&nbsp;MB';
    } elseif( $kiloBytes >= 0 ) {
        return number_format_i18n( $kiloBytes/1, 0 ).'&nbsp;KB';
    } else {
        return 'Unknown';
    }
  }

	public function set_wp_content_url(){
		$content_url = content_url();

		// Hide My WP
		if(in_array( 'hide_my_wp/hide-my-wp.php', (array) get_option( 'active_plugins', array() ) )){
			$hide_my_wp = get_option("hide_my_wp");

			if(isset($hide_my_wp["new_content_path"]) && $hide_my_wp["new_content_path"]){
				$hide_my_wp["new_content_path"] = trim($hide_my_wp["new_content_path"], "/");
				$content_url = str_replace(basename(WP_CONTENT_DIR), $hide_my_wp["new_content_path"], $content_url);
			}
		}

		if (!defined('CYBVC_WP_CONTENT_URL')) {
			define("CYBVC_WP_CONTENT_URL", $content_url);
		}
	}

  public static function initiate_cron() {
    $args = array(
      'timeout'    => 1,
      'blocking'   => false,
      'sslverify'  => false,
    );
    $url = get_site_url();
    wp_remote_get( $url . "/wp-cron.php?doing_wp_cron", $args);
  }


  public static function _extra_preconnect($hints, $relation_type) {

    // if ( 'prefetch' === $relation_type  ) {
    //   $hints = array_merge($hints, array('//www.google-analytics.com'));
    // }
    // else
    if ( 'preconnect' === $relation_type  ) {
        $_preconn_hint = array('href' => 'https://fonts.gstatic.com/');
        $_preconn_hint['crossorigin'] = 'anonymous';
        $hints[] = $_preconn_hint;
    }

    return $hints;
  }

  /**
   * Get all uri we don't cache
   *
   * @since 2.0
   *
   * @return array List of uri not to cache
   */
  public static function get_reject_uri($url = array()) {

  	$uri = get_option( 'viper_cache_reject_uri', array() );

  	$uri = apply_filters( 'viper_cache_reject_uri', $uri );

    $uri = array_unique (array_merge( $uri, $url ));

  	$uri = implode( '|', array_filter( $uri ) );

    return $uri;
  }

  /**
   * Get all JS files we don't cache / minimise / combine
   *
   * @since 2.0
   *
   * @return array List of JS Files not to cache
   */
  public static function get_excluded_js_files($JSFiles = array()) {

  	$excludedJS = apply_filters( 'viper_cache_excluded_js_files', $JSFiles);

    $excludedJS = array_unique (array_merge( $excludedJS, $JSFiles ));

  	$excludedJS = implode( '|', array_filter( $excludedJS ) );

    return $excludedJS;
  }

  /**
   * Update the list of 3rd party rejected uri we don't cache
   *
   * @since 2.0
   *
   * @return array List of uri not to cache
   */
  public static function update_3rdparty_reject_uri($uri = array()) {

  	$uri = apply_filters( 'viper_cache_reject_uri', $uri );

    update_option( 'viper_cache_reject_uri', $uri );

  }

  /**
   * Update the list of JS files we don't cache with users settings
   *
   * @since 2.0
   *
   * @return array List of JS files not to cache
   */
  public static function update_excluded_js_files($JSFiles = array()) {

			$JSFiles = array_unique (array_merge( $JSFiles, self::$cacheSettings['doNotCacheJS'] ));

      return $JSFiles;
  }

  /**
   * Get the url path of a specific post with ID.
   *
   * @since   2.0
   *
   * @param   int    $post_id    Post ID.
   * @param   string $post_type  Post Type.
   * @param   string $regex      Regex to include at the end.
   * @return  array   $urls
   */
  public static function get_post_url_path( $post_id, $post_type = 'page', $regex = null ) {

  	$urls  = array();
    $url = get_permalink( $post_id );

    $path = _get_component_from_parsed_url_array( wp_parse_url( $url ), PHP_URL_PATH );

  	if ( empty( $path ) ) {
  		return $urls;
  	}

  	if ( trim( $path, '/' ) !== '' ) {
  		$urls[] = $path . $regex;
  	}

  	$urls = array_unique( $urls );

  	return $urls;
  }

}
