<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://vipercache.com
 * @since      1.0.0
 *
 * @package    ViperCache
 * @subpackage ViperCache/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    ViperCache
 * @subpackage ViperCache/includes
 * @author     Your Name <email@example.com>
 */
class ViperCache {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      ViperCache_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		$this->plugin_name = 'vipercache';
		$this->version = '2.1';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
    $this->define_metabox_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - ViperCache_Loader. Orchestrates the hooks of the plugin.
	 * - ViperCache_i18n. Defines internationalization functionality.
	 * - ViperCache_Admin. Defines all hooks for the admin area.
	 * - ViperCache_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-vipercache-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-vipercache-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-vipercache-settings.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-vipercache-admin.php';

		/**
 		 * The class responsible for defining all actions relating to metaboxes.
 		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-vipercache-admin-metaboxes.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-vipercache-public.php';

		/**
		 * The class responsible for providing the site caching
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-viper-cache.php';

		/**
		 * The class responsible for the background processing of the plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/wp-background-processing.php';

		/**
		 * The class responsible for dealing with licensing the plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/DigitalGuard_IT_Licensing_2_3.php';

		/**
		 * The class responsible for dealing with auto updates for the plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/DigitalGuard_IT_Updater_1_1.php';

		$this->loader = new ViperCache_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the ViperCache_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new ViperCache_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new ViperCache_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
    $this->loader->add_action( 'admin_head', $plugin_admin, 'admin_head');
    $this->loader->add_action( 'wp_ajax_' . CYBVC_SLUG . '_clear_cache', $plugin_admin, 'clear_cache');
    $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_menu' );
    $this->loader->add_action( 'admin_bar_menu', $plugin_admin, 'admin_bar_menu', 1000 );
    $this->loader->add_action( 'plugins_loaded', $plugin_admin, 'load_third_party_dependencies' );
    $this->loader->add_action( 'admin_init', $plugin_admin, 'regenerate_3rdparty_reject_uri' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new ViperCache_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
    // Setup filers for when cache is active

    if($plugin_public->sitecache_settings['active'] == 'Y') {

      if($plugin_public->sitecache_settings['deferJS'] == 'Y')
        $this->loader->add_filter( 'script_loader_tag',  $plugin_public, 'defer_parsing_of_js', 10, 3 );

      if($plugin_public->sitecache_settings['disableEmojis'] == 'Y')
				$this->loader->add_action('init', $plugin_public, 'disable_emojis');

      if($plugin_public->sitecache_settings['removeVerQS'] == 'Y') {
        $this->loader->add_filter( 'script_loader_src', $plugin_public, 'remove_version_from_query_string' );
        $this->loader->add_filter( 'style_loader_src', $plugin_public, 'remove_version_from_query_string' );
      }

		}

	}

  /**
	 * Register all of the hooks related to metaboxes
	 *
	 * @since 		1.0.0
	 * @access 		private
	 */
	private function define_metabox_hooks() {

		$plugin_metaboxes = new ViperCache_Admin_Metaboxes( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'add_meta_boxes', $plugin_metaboxes, 'add_metaboxes' );
		$this->loader->add_action( 'save_post', $plugin_metaboxes, 'validate_meta', 10, 2 );

	} // define_metabox_hooks()

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    ViperCache_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
